<?php
// توابع مدیریت تخفیف خرید مجدد
require_once __DIR__ . '/../conf/config.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/message_utils.php';
require_once __DIR__ . '/user_order_utils.php';
require_once __DIR__ . '/product_utils.php';

// ایجاد جدول در صورت عدم وجود
function initRepeatPurchaseDiscountTable() {
    $sql = "CREATE TABLE IF NOT EXISTS repeat_purchase_discounts (
        id INT AUTO_INCREMENT PRIMARY KEY,
        product_id INT NOT NULL,
        variation_id INT DEFAULT 0,
        discount_percent INT DEFAULT 0,
        is_active TINYINT(1) DEFAULT 0,
        min_purchase_count INT DEFAULT 1,
        min_days_since_last_purchase INT DEFAULT 0,
        custom_message TEXT,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY unique_product_variation (product_id, variation_id),
        INDEX idx_product (product_id),
        INDEX idx_active (is_active)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    try {
        dbQuery($sql);
    } catch (Exception $e) {
        error_log("Error creating repeat_purchase_discounts table: " . $e->getMessage());
    }
}

// دریافت تنظیمات تخفیف برای یک محصول/ویژگی
function getRepeatPurchaseDiscount($product_id, $variation_id = 0) {
    $product_id = (int)$product_id;
    $variation_id = (int)$variation_id;
    $stmt = dbQuery("SELECT * FROM repeat_purchase_discounts WHERE product_id = ? AND variation_id = ?", [$product_id, $variation_id]);
    $result = $stmt->fetch();
    // تبدیل is_active به integer برای اطمینان
    if($result && isset($result['is_active'])) {
        $result['is_active'] = (int)$result['is_active'];
    }
    return $result;
}

// دریافت تمام تخفیف‌های فعال
function getAllRepeatPurchaseDiscounts() {
    $stmt = dbQuery("SELECT * FROM repeat_purchase_discounts ORDER BY product_id, variation_id");
    return $stmt->fetchAll();
}

// ذخیره یا به‌روزرسانی تخفیف
function saveRepeatPurchaseDiscount($product_id, $variation_id, $discount_percent, $is_active, $min_purchase_count, $min_days_since_last_purchase, $custom_message) {
    // تبدیل به integer برای اطمینان
    $product_id = (int)$product_id;
    $variation_id = (int)$variation_id;
    $discount_percent = (int)$discount_percent;
    $is_active = (int)$is_active;
    $min_purchase_count = (int)$min_purchase_count;
    $min_days_since_last_purchase = (int)$min_days_since_last_purchase;
    
    $stmt = dbQuery("SELECT id FROM repeat_purchase_discounts WHERE product_id = ? AND variation_id = ?", [$product_id, $variation_id]);
    $existing = $stmt->fetch();
    
    if($existing) {
        dbQuery("UPDATE repeat_purchase_discounts SET discount_percent = ?, is_active = ?, min_purchase_count = ?, min_days_since_last_purchase = ?, custom_message = ? WHERE id = ?", 
            [$discount_percent, $is_active, $min_purchase_count, $min_days_since_last_purchase, $custom_message, $existing['id']]);
    } else {
        dbQuery("INSERT INTO repeat_purchase_discounts (product_id, variation_id, discount_percent, is_active, min_purchase_count, min_days_since_last_purchase, custom_message) VALUES (?, ?, ?, ?, ?, ?, ?)",
            [$product_id, $variation_id, $discount_percent, $is_active, $min_purchase_count, $min_days_since_last_purchase, $custom_message]);
    }
}

// بررسی اینکه آیا کاربر شرایط تخفیف را دارد
function checkUserEligibleForDiscount($user_mobile, $product_id, $variation_id = 0) {
    $discount = getRepeatPurchaseDiscount($product_id, $variation_id);
    
    if(!$discount) {
        return ['eligible' => false];
    }
    
    // تبدیل is_active به integer برای مقایسه صحیح
    $is_active = (int)$discount['is_active'];
    if($is_active != 1) {
        return ['eligible' => false];
    }
    
    $mobile_norm = normalizeMobile($user_mobile);
    if(!$mobile_norm || strlen($mobile_norm) !== 11) {
        return ['eligible' => false];
    }
    
    // بررسی تعداد خریدهای قبلی
    // دریافت تمام سفارش‌های کاربر (تمام وضعیت‌ها) با pagination
    $user_orders = [];
    $page = 1;
    $per_page = 100;
    
    // بررسی همه صفحات تا زمانی که سفارش‌ها تمام شوند
    while(true) {
        $url = "https://goldencheat.ir/wp-json/wc/v3/orders?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET . "&per_page={$per_page}&page={$page}";
        $response = @file_get_contents($url);
        if($response === false) {
            break; // در صورت خطا، ادامه نده
        }
        $orders = json_decode($response, true);
        
        if(!is_array($orders) || empty($orders)) {
            break; // اگر صفحه خالی بود، یعنی تمام شد
        }
        
        // فیلتر سفارش‌های کاربر بر اساس شماره موبایل
        foreach($orders as $order) {
            $order_mobile = normalizeMobile($order['billing']['phone'] ?? '');
            if($order_mobile && strlen($order_mobile) === 11 && $order_mobile == $mobile_norm) {
                $user_orders[] = $order;
            }
        }
        
        // اگر سفارش‌های این صفحه کمتر از per_page بود، یعنی آخرین صفحه است
        if(count($orders) < $per_page) {
            break;
        }
        
        $page++;
    }
    
    $purchase_count = 0;
    $last_purchase_date = null;
    
    // تبدیل به integer برای مقایسه صحیح
    $product_id = (int)$product_id;
    $variation_id = (int)$variation_id;
    
    foreach($user_orders as $order) {
        foreach($order['line_items'] as $item) {
            $item_product_id = (int)($item['product_id'] ?? 0);
            $item_variation_id = (int)($item['variation_id'] ?? 0);
            
            // بررسی تطابق محصول
            if($item_product_id == $product_id) {
                // اگر variation_id = 0 باشد، یعنی تخفیف برای کل محصول است
                // در غیر این صورت باید variation_id هم تطابق داشته باشد
                if($variation_id == 0 || $item_variation_id == $variation_id) {
                    $purchase_count++;
                    $order_date = strtotime($order['date_created'] ?? $order['date_completed'] ?? '');
                    if($order_date && (!$last_purchase_date || $order_date > $last_purchase_date)) {
                        $last_purchase_date = $order_date;
                    }
                }
            }
        }
    }
    
    // لاگ برای دیباگ (فقط در صورت نیاز)
    // error_log("Discount check: mobile={$mobile_norm}, product_id={$product_id}, variation_id={$variation_id}, found_orders=" . count($user_orders) . ", purchase_count={$purchase_count}, min_required={$discount['min_purchase_count']}");
    
    // بررسی شرایط
    if($purchase_count < $discount['min_purchase_count']) {
        return ['eligible' => false];
    }
    
    if($discount['min_days_since_last_purchase'] > 0 && $last_purchase_date) {
        $days_since = floor((time() - $last_purchase_date) / 86400);
        if($days_since < $discount['min_days_since_last_purchase']) {
            return ['eligible' => false];
        }
    }
    
    return [
        'eligible' => true,
        'discount' => $discount,
        'purchase_count' => $purchase_count,
        'last_purchase_date' => $last_purchase_date
    ];
}

// محاسبه قیمت با تخفیف
function calculateDiscountedPrice($original_price, $discount_percent) {
    $discount_amount = ($original_price * $discount_percent) / 100;
    return $original_price - $discount_amount;
}

// نمایش لیست محصولات با وضعیت تخفیف
function showRepeatPurchaseDiscountProductsList($chat_id, $message_id = null) {
    $products = getProductsFromApi();
    $discounts = getAllRepeatPurchaseDiscounts();
    
    // ساخت آرایه سریع برای جستجو
    $discount_map = [];
    foreach($discounts as $d) {
        $key = $d['product_id'] . '_' . $d['variation_id'];
        $discount_map[$key] = $d;
    }
    
    $text = "🎁 <b>تخفیف خرید مجدد</b>\n\n";
    $text .= "لیست محصولات:\n\n";
    
    $inline_keyboard = [];
    foreach($products as $p) {
        $product_id = $p['id'];
        $product_name = $p['name'];
        
        // بررسی وجود تخفیف فعال
        $has_active_discount = false;
        foreach($discounts as $d) {
            if($d['product_id'] == $product_id && $d['is_active']) {
                $has_active_discount = true;
                break;
            }
        }
        
        $status_icon = $has_active_discount ? '✅' : '❌';
        $text .= "{$status_icon} {$product_name}\n";
        
        $inline_keyboard[] = [
            ['text' => "{$status_icon} {$product_name}", 'callback_data' => 'rpd_product_' . $product_id]
        ];
    }
    
    $inline_keyboard[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_panel']];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    
    if($message_id) {
        editMessage($chat_id, $message_id, $text, $reply_markup);
    } else {
        sendMessage($chat_id, $text, $reply_markup);
    }
}

// نمایش جزئیات تخفیف یک محصول
function showRepeatPurchaseDiscountProductDetails($chat_id, $product_id, $message_id = null) {
    $products = getProductsFromApi();
    $product = null;
    foreach($products as $p) {
        if($p['id'] == $product_id) {
            $product = $p;
            break;
        }
    }
    
    if(!$product) {
        sendMessage($chat_id, 'محصول یافت نشد.');
        return;
    }
    
    $variations = getProductVariationsFromApi($product_id);
    $discounts = getAllRepeatPurchaseDiscounts();
    
    $text = "🎁 <b>تخفیف خرید مجدد</b>\n\n";
    $text .= "📦 <b>محصول:</b> {$product['name']}\n";
    $text .= "━━━━━━━━━━━━━━━━━━\n\n";
    
    $inline_keyboard = [];
    
    if(is_array($variations) && count($variations) > 0) {
        $text .= "<b>ویژگی‌های محصول:</b>\n\n";
        
        foreach($variations as $var) {
            $var_id = $var['id'] ?? 0;
            $var_price = $var['price'] ?? 0;
            
            // ساخت نام ویژگی
            $var_name_parts = [];
            if(isset($var['attributes']) && is_array($var['attributes'])) {
                foreach($var['attributes'] as $attr) {
                    $var_name_parts[] = ($attr['name'] ?? '') . ': ' . ($attr['option'] ?? '');
                }
            }
            $var_name = !empty($var_name_parts) ? implode(' - ', $var_name_parts) : "ویژگی #{$var_id}";
            
            // بررسی تخفیف
            $discount = null;
            foreach($discounts as $d) {
                if($d['product_id'] == $product_id && $d['variation_id'] == $var_id) {
                    $discount = $d;
                    break;
                }
            }
            
            $status_icon = ($discount && $discount['is_active']) ? '✅' : '❌';
            $discount_text = '';
            $discounted_price = $var_price;
            
            if($discount && $discount['is_active']) {
                $discounted_price = calculateDiscountedPrice($var_price, $discount['discount_percent']);
                $discount_text = "\n   💰 قیمت اصلی: " . number_format($var_price) . " تومان";
                $discount_text .= "\n   🎁 قیمت با تخفیف ({$discount['discount_percent']}%): " . number_format($discounted_price) . " تومان";
            } else {
                $discount_text = "\n   💰 قیمت: " . number_format($var_price) . " تومان";
            }
            
            $text .= "{$status_icon} <b>{$var_name}</b>{$discount_text}\n\n";
            
            $inline_keyboard[] = [
                ['text' => "⚙️ تنظیمات {$var_name}", 'callback_data' => 'rpd_edit_' . $product_id . '_' . $var_id]
            ];
        }
    } else {
        // محصول ساده بدون ویژگی
        $product_price = $product['price'] ?? 0;
        $discount = null;
        foreach($discounts as $d) {
            if($d['product_id'] == $product_id && $d['variation_id'] == 0) {
                $discount = $d;
                break;
            }
        }
        
        $status_icon = ($discount && $discount['is_active']) ? '✅' : '❌';
        $discount_text = '';
        $discounted_price = $product_price;
        
        if($discount && $discount['is_active']) {
            $discounted_price = calculateDiscountedPrice($product_price, $discount['discount_percent']);
            $discount_text = "\n💰 قیمت اصلی: " . number_format($product_price) . " تومان";
            $discount_text .= "\n🎁 قیمت با تخفیف ({$discount['discount_percent']}%): " . number_format($discounted_price) . " تومان";
        } else {
            $discount_text = "\n💰 قیمت: " . number_format($product_price) . " تومان";
        }
        
        $text .= "{$status_icon} <b>محصول ساده</b>{$discount_text}\n\n";
        
        $inline_keyboard[] = [
            ['text' => '⚙️ تنظیمات محصول', 'callback_data' => 'rpd_edit_' . $product_id . '_0']
        ];
    }
    
    $inline_keyboard[] = [['text' => '🔙 بازگشت', 'callback_data' => 'rpd_list']];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    
    if($message_id) {
        editMessage($chat_id, $message_id, $text, $reply_markup);
    } else {
        sendMessage($chat_id, $text, $reply_markup);
    }
}

// نمایش صفحه تنظیمات تخفیف
function showRepeatPurchaseDiscountSettings($chat_id, $product_id, $variation_id, $message_id = null) {
    $products = getProductsFromApi();
    $product = null;
    foreach($products as $p) {
        if($p['id'] == $product_id) {
            $product = $p;
            break;
        }
    }
    
    if(!$product) {
        sendMessage($chat_id, 'محصول یافت نشد.');
        return;
    }
    
    $discount = getRepeatPurchaseDiscount($product_id, $variation_id);
    
    $variation_name = 'محصول ساده';
    if($variation_id > 0) {
        $variations = getProductVariationsFromApi($product_id);
        foreach($variations as $var) {
            if(($var['id'] ?? 0) == $variation_id) {
                $var_name_parts = [];
                if(isset($var['attributes']) && is_array($var['attributes'])) {
                    foreach($var['attributes'] as $attr) {
                        $var_name_parts[] = ($attr['name'] ?? '') . ': ' . ($attr['option'] ?? '');
                    }
                }
                $variation_name = !empty($var_name_parts) ? implode(' - ', $var_name_parts) : "ویژگی #{$variation_id}";
                break;
            }
        }
    }
    
    $text = "⚙️ <b>تنظیمات تخفیف خرید مجدد</b>\n\n";
    $text .= "📦 محصول: {$product['name']}\n";
    $text .= "🔖 ویژگی: {$variation_name}\n";
    $text .= "━━━━━━━━━━━━━━━━━━\n\n";
    
    // خواندن مجدد از دیتابیس برای اطمینان از داده‌های به‌روز
    $discount = getRepeatPurchaseDiscount($product_id, $variation_id);
    
    if($discount) {
        $is_active = (int)$discount['is_active'];
        $status_text = ($is_active == 1) ? '✅ فعال' : '❌ غیرفعال';
        $text .= "<b>وضعیت فعلی:</b> {$status_text}\n";
        $text .= "<b>درصد تخفیف:</b> {$discount['discount_percent']}%\n";
        $text .= "<b>حداقل تعداد خرید:</b> {$discount['min_purchase_count']}\n";
        $text .= "<b>حداقل روز از آخرین خرید:</b> {$discount['min_days_since_last_purchase']}\n";
        if($discount['custom_message']) {
            $text .= "\n<b>پیام سفارشی:</b>\n" . mb_substr($discount['custom_message'], 0, 100) . "...\n";
        }
    } else {
        $text .= "❌ هنوز تنظیم نشده است.\n";
        $is_active = 0;
    }
    
    $text .= "\n━━━━━━━━━━━━━━━━━━\n";
    $text .= "برای تغییر تنظیمات، یکی از گزینه‌های زیر را انتخاب کنید:";
    
    $inline_keyboard = [
        [
            ['text' => ($is_active == 1) ? '❌ غیرفعال کردن' : '✅ فعال کردن', 'callback_data' => 'rpd_toggle_' . $product_id . '_' . $variation_id]
        ],
        [
            ['text' => '📊 تنظیم درصد تخفیف', 'callback_data' => 'rpd_setpercent_' . $product_id . '_' . $variation_id]
        ],
        [
            ['text' => '🔢 تنظیم حداقل خرید', 'callback_data' => 'rpd_setminpurchase_' . $product_id . '_' . $variation_id]
        ],
        [
            ['text' => '📅 تنظیم حداقل روز', 'callback_data' => 'rpd_setmindays_' . $product_id . '_' . $variation_id]
        ],
        [
            ['text' => '💬 تنظیم پیام سفارشی', 'callback_data' => 'rpd_setmessage_' . $product_id . '_' . $variation_id]
        ],
        [
            ['text' => '🔙 بازگشت', 'callback_data' => 'rpd_product_' . $product_id]
        ]
    ];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    
    if($message_id) {
        editMessage($chat_id, $message_id, $text, $reply_markup);
    } else {
        sendMessage($chat_id, $text, $reply_markup);
    }
}

// ارسال پیام اطلاع‌رسانی به کاربران واجد شرایط
function notifyEligibleUsersForDiscount($product_id, $variation_id) {
    $discount = getRepeatPurchaseDiscount($product_id, $variation_id);
    if(!$discount || !$discount['is_active']) {
        return ['sent' => 0, 'error' => 0, 'total' => 0];
    }
    
    // دریافت اطلاعات محصول
    $products = getProductsFromApi();
    $product_name = '';
    foreach($products as $p) {
        if($p['id'] == $product_id) {
            $product_name = $p['name'];
            break;
        }
    }
    
    if(!$product_name) {
        return ['sent' => 0, 'error' => 0, 'total' => 0];
    }
    
    // دریافت تمام سفارش‌ها (با pagination) - یک بار
    $all_orders = [];
    $page = 1;
    $per_page = 100;
    
    do {
        $url = "https://goldencheat.ir/wp-json/wc/v3/orders?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET . "&per_page={$per_page}&page={$page}";
        $response = @file_get_contents($url);
        $orders = json_decode($response, true);
        
        if(!is_array($orders) || empty($orders)) {
            break;
        }
        
        $all_orders = array_merge($all_orders, $orders);
        $page++;
        // محدود کردن به 20 صفحه برای جلوگیری از timeout
        if($page > 20) break;
    } while(count($orders) == $per_page);
    
    if(empty($all_orders)) {
        return ['sent' => 0, 'error' => 0, 'total' => 0];
    }
    
    // ساخت map از شماره موبایل به لیست سفارش‌ها
    $mobile_to_orders = [];
    foreach($all_orders as $order) {
        $order_mobile = normalizeMobile($order['billing']['phone'] ?? '');
        if($order_mobile && strlen($order_mobile) === 11) {
            if(!isset($mobile_to_orders[$order_mobile])) {
                $mobile_to_orders[$order_mobile] = [];
            }
            $mobile_to_orders[$order_mobile][] = $order;
        }
    }
    
    // دریافت تمام کاربران ربات با شماره موبایل
    $stmt = dbQuery("SELECT telegram_id, mobile FROM users WHERE mobile IS NOT NULL AND mobile != '' AND LENGTH(mobile) = 11");
    $all_users = $stmt->fetchAll();
    
    // ساخت لیست کاربران واجد شرایط
    $eligible_users = [];
    
    foreach($all_users as $user) {
        $user_mobile = normalizeMobile($user['mobile']);
        if(!$user_mobile || strlen($user_mobile) !== 11) {
            continue;
        }
        
        // بررسی اینکه آیا این کاربر سفارشی برای این محصول دارد
        if(!isset($mobile_to_orders[$user_mobile])) {
            continue;
        }
        
        $user_orders = $mobile_to_orders[$user_mobile];
        $purchase_count = 0;
        $last_purchase_date = null;
        
        // تبدیل به integer برای مقایسه صحیح
        $product_id_int = (int)$product_id;
        $variation_id_int = (int)$variation_id;
        
        // شمارش خریدهای این محصول
        foreach($user_orders as $order) {
            foreach($order['line_items'] as $item) {
                $item_product_id = (int)($item['product_id'] ?? 0);
                $item_variation_id = (int)($item['variation_id'] ?? 0);
                
                // بررسی تطابق محصول
                if($item_product_id == $product_id_int) {
                    // اگر variation_id = 0 باشد، یعنی تخفیف برای کل محصول است
                    // در غیر این صورت باید variation_id هم تطابق داشته باشد
                    if($variation_id_int == 0 || $item_variation_id == $variation_id_int) {
                        $purchase_count++;
                        $order_date = strtotime($order['date_created'] ?? $order['date_completed'] ?? '');
                        if($order_date && (!$last_purchase_date || $order_date > $last_purchase_date)) {
                            $last_purchase_date = $order_date;
                        }
                    }
                }
            }
        }
        
        // بررسی شرایط
        if($purchase_count < $discount['min_purchase_count']) {
            continue;
        }
        
        if($discount['min_days_since_last_purchase'] > 0 && $last_purchase_date) {
            $days_since = floor((time() - $last_purchase_date) / 86400);
            if($days_since < $discount['min_days_since_last_purchase']) {
                continue;
            }
        }
        
        // کاربر واجد شرایط است
        $telegram_id = $user['telegram_id'];
        if(!isset($eligible_users[$telegram_id])) {
            $eligible_users[$telegram_id] = true;
        }
    }
    
    // ارسال پیام به کاربران واجد شرایط
    $message = "🔥🎉 <b>خبر فوق‌العاده برای شما!</b> 🎉🔥\n\n";
    $message .= "به پاس اینکه قبــلاً از GoldenCheat خرید داشتین،\n\n";
    $message .= "روی محصول «<b>{$product_name}</b>» همین الان یه تخفیف خفن و ویژه برای خرید مجدد فعال شده! 😎💛\n\n";
    $message .= "💥 <b>میزان تخفیف شما: {$discount['discount_percent']} درصد</b> 💥\n\n";
    $message .= "این امتیاز فقط برای مشتری‌های وفادار فعاله و شامل همه نمی‌شه!\n\n";
    $message .= "✨ همین الان می‌تونی با قیمت جدید و تخفیف‌خورده، سفارشت رو ثبت کنی.\n\n";
    $message .= "عجله کن، این فرصت همیشه باز نیست! 🚀🔥";
    
    $sent_count = 0;
    $error_count = 0;
    $error_details = [];
    
    foreach(array_keys($eligible_users) as $telegram_id) {
        $result = sendMessage($telegram_id, $message);
        if($result) {
            $sent_count++;
        } else {
            $error_count++;
            // دریافت اطلاعات کاربر برای لاگ خطا
            $stmt = dbQuery("SELECT mobile FROM users WHERE telegram_id = ?", [$telegram_id]);
            $user = $stmt->fetch();
            $mobile = $user ? $user['mobile'] : 'نامشخص';
            $error_details[] = "کاربر {$telegram_id} ({$mobile})";
            
            // لاگ خطا
            error_log("Failed to send discount notification to user {$telegram_id} (mobile: {$mobile})");
        }
        // تاخیر برای جلوگیری از rate limit
        usleep(500000); // 0.5 ثانیه
    }
    
    return [
        'sent' => $sent_count,
        'error' => $error_count,
        'total' => count($eligible_users),
        'error_details' => $error_details
    ];
}

// مقداردهی اولیه جدول
initRepeatPurchaseDiscountTable();

