<?php
// توابع مدیریت کاربران و سفارشات
require_once __DIR__ . '/../conf/config.php';
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/message_utils.php';

if(!defined('API_URL')) define('API_URL', 'https://api.telegram.org/bot' . BOT_TOKEN . '/');

// لاگ خطا فقط در صورت نیاز (برای کاهش حجم فایل‌های log)
// file_put_contents('cache_error.log', date('Y-m-d H:i:s')." test log\n", FILE_APPEND);

function normalizeMobile($mobile) {
    $mobile = trim($mobile);
    $mobile = str_replace([' ', '-', '(', ')'], '', $mobile);
    if(preg_match('/^(\+98|0098|098|98)/', $mobile)) $mobile = preg_replace('/^(\+98|0098|098|98)/', '0', $mobile);
    if(preg_match('/^9\d{9}$/', $mobile)) $mobile = '0' . $mobile;
    $mobile = preg_replace('/[^0-9]/', '', $mobile);
    if(strlen($mobile) != 11 || strpos($mobile, '09') !== 0) return $mobile;
    return $mobile;
}
function registerUser($user_id, $mobile = null) {
    if($mobile) {
        $mobile = normalizeMobile($mobile);
        $stmt = dbQuery("SELECT id FROM users WHERE telegram_id = ?", [$user_id]);
        if($stmt->fetch()) {
            dbQuery("UPDATE users SET created_at = NOW(), mobile = ? WHERE telegram_id = ?", [$mobile, $user_id]);
        } else {
            dbQuery("INSERT INTO users (telegram_id, mobile) VALUES (?, ?)", [$user_id, $mobile]);
        }
    } else {
        $stmt = dbQuery("SELECT id FROM users WHERE telegram_id = ?", [$user_id]);
        if(!$stmt->fetch()) {
            dbQuery("INSERT INTO users (telegram_id) VALUES (?)", [$user_id]);
        }
    }
}
function getUserStats() {
    $today = date('Y-m-d');
    $month = date('Y-m');
    $total = dbQuery("SELECT COUNT(*) as cnt FROM users")->fetch()['cnt'];
    $today_count = dbQuery("SELECT COUNT(*) as cnt FROM users WHERE DATE(created_at) = ?", [$today])->fetch()['cnt'];
    $month_count = dbQuery("SELECT COUNT(*) as cnt FROM users WHERE DATE_FORMAT(created_at, '%Y-%m') = ?", [$month])->fetch()['cnt'];
    return [
        'total' => $total,
        'today' => $today_count,
        'month' => $month_count
    ];
}
function getOrderStatsFromApi() {
    $url = "https://goldencheat.ir/wp-json/wc/v3/orders?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET;
    $response = @file_get_contents($url);
    $orders = json_decode($response, true);
    return is_array($orders) ? count($orders) : 0;
}
function getOrderStatsFromDb() {
    $total = dbQuery("SELECT COUNT(*) as cnt FROM orders")->fetch()['cnt'];
    return $total;
}
function getProductsFromApi() {
    $cache_file = __DIR__ . '/products_cache.json';
    $cache_time = 18000; // ۵ ساعت

    if(file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
        $products = json_decode(file_get_contents($cache_file), true);
        if(is_array($products) && count($products) > 0) return $products;
        // اگر کش خراب یا خالی است، پاک کن و لاگ کن
        @unlink($cache_file);
        file_put_contents(__DIR__ . '/cache_error.log', date('Y-m-d H:i:s')." products_cache.json is empty or invalid\n", FILE_APPEND);
    }

    $url = WC_API_URL . "?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET . "&per_page=100";
    $response = @file_get_contents($url);
    $products = json_decode($response, true);
    if($products && is_array($products) && count($products) > 0) {
        file_put_contents($cache_file, json_encode($products));
        return $products;
    }
    // اگر کش قبلی وجود دارد و سالم نیست، پاک کن و لاگ کن
    if(file_exists($cache_file)) {
        $products = json_decode(file_get_contents($cache_file), true);
        if(!is_array($products) || count($products) == 0) {
            @unlink($cache_file);
            file_put_contents(__DIR__ . '/cache_error.log', date('Y-m-d H:i:s')." products_cache.json is empty or invalid\n", FILE_APPEND);
        }
    }
    return [];
}

// کش متغیرهای هر محصول
function getProductVariationsFromApi($product_id) {
    $cache_file = __DIR__ . "/product_vars_cache_{$product_id}.json";
    $cache_time = 18000;
    if(file_exists($cache_file) && (time() - filemtime($cache_file)) < $cache_time) {
        $vars = json_decode(file_get_contents($cache_file), true);
        if(is_array($vars) && count($vars) > 0) return $vars;
        // اگر کش خراب یا خالی است، پاک کن و لاگ کن
        @unlink($cache_file);
        file_put_contents(__DIR__ . '/cache_error.log', date('Y-m-d H:i:s')." product_vars_cache_{$product_id}.json is empty or invalid\n", FILE_APPEND);
    }
    $url = WC_API_URL . "/$product_id/variations?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET;
    $response = @file_get_contents($url);
    $vars = json_decode($response, true);
    if($vars && is_array($vars) && count($vars) > 0) {
        file_put_contents($cache_file, json_encode($vars));
        return $vars;
    }
    if(file_exists($cache_file)) {
        $vars = json_decode(file_get_contents($cache_file), true);
        if(!is_array($vars) || count($vars) == 0) {
            @unlink($cache_file);
            file_put_contents(__DIR__ . '/cache_error.log', date('Y-m-d H:i:s')." product_vars_cache_{$product_id}.json is empty or invalid\n", FILE_APPEND);
        }
    }
    return [];
}

function showStartMenu($chat_id, $is_admin = false, $message_id = null) {
    global $user_id;
    registerUser($user_id);
    $reply_keyboard = [
        'keyboard' => [
            [['text' => '🛒 خرید چیت']],
            [['text' => '🔑 لایسنس های من']],
            [['text' => '🎓 دریافت آموزش من']],
            [['text' => '📦 سفارشات من']],
            $is_admin ? [['text' => '🛠 پنل مدیریت']] : [],
        ],
        'resize_keyboard' => true,
        'one_time_keyboard' => false
    ];
    $text = "🎬 به مرکز آموزش گلدن چیت خوش آمدید!";
    if($message_id)
        editMessage($chat_id, $message_id, $text, $reply_keyboard);
    else
        sendMessage($chat_id, $text, $reply_keyboard);
}
// توابع امن برای مدیریت فایل‌های موقت
function sanitizeFileName($name) {
    // حذف کاراکترهای خطرناک و محدود کردن به حروف، اعداد و خط تیره
    $name = preg_replace('/[^a-zA-Z0-9_-]/', '', $name);
    // محدود کردن طول نام فایل
    return substr($name, 0, 100);
}

function safeFilePutContents($filename, $data) {
    // اعتبارسنجی نام فایل
    $filename = sanitizeFileName($filename);
    // اطمینان از اینکه فقط فایل‌های .tmp در root نوشته می‌شوند
    if(strpos($filename, '..') !== false || strpos($filename, '/') !== false || strpos($filename, '\\') !== false) {
        error_log("Security: Attempted path traversal in filename: " . $filename);
        return false;
    }
    // فقط فایل‌های .tmp مجاز هستند
    if(substr($filename, -4) !== '.tmp') {
        error_log("Security: Invalid file extension: " . $filename);
        return false;
    }
    return @file_put_contents($filename, $data);
}

function safeFileGetContents($filename) {
    // اعتبارسنجی نام فایل
    $filename = sanitizeFileName($filename);
    // اطمینان از اینکه فقط فایل‌های .tmp خوانده می‌شوند
    if(strpos($filename, '..') !== false || strpos($filename, '/') !== false || strpos($filename, '\\') !== false) {
        error_log("Security: Attempted path traversal in filename: " . $filename);
        return false;
    }
    // فقط فایل‌های .tmp مجاز هستند
    if(substr($filename, -4) !== '.tmp') {
        error_log("Security: Invalid file extension: " . $filename);
        return false;
    }
    return @file_get_contents($filename);
}

function setTempOrderId($user_id, $order_id) { 
    // اعتبارسنجی ورودی
    $user_id = (int)$user_id;
    $order_id = preg_replace('/[^0-9]/', '', $order_id);
    safeFilePutContents("order_{$user_id}.tmp", $order_id); 
}

function getTempOrderId($user_id) { 
    // اعتبارسنجی ورودی
    $user_id = (int)$user_id;
    $order_id = safeFileGetContents("order_{$user_id}.tmp"); 
    if($order_id) @unlink("order_{$user_id}.tmp"); 
    return $order_id; 
}
function getOrderFromApi($order_id) {
    $url = "https://goldencheat.ir/wp-json/wc/v3/orders/{$order_id}?consumer_key=" . WC_CONSUMER_KEY . "&consumer_secret=" . WC_CONSUMER_SECRET;
    $response = @file_get_contents($url);
    if(!$response) return null;
    $order = json_decode($response, true);
    return $order;
}
function sendProductTraining($chat_id, $order_id, $mobile) {
    $mobile = normalizeMobile($mobile);
    $order = getOrderFromApi($order_id);
    if(!$order) {
        sendMessage($chat_id, '❗ سفارش شما پیدا نشد. لطفاً دقت کنید یا با پشتیبانی تماس بگیرید.');
        return;
    }
    // فقط سفارش‌های تکمیل شده یا در حال انجام
    $status = $order['status'] ?? '';
    if($status !== 'completed' && $status !== 'processing') {
        sendMessage($chat_id, '❗ فقط سفارش‌های تکمیل شده یا در حال انجام مجاز به دریافت آموزش هستند.');
        return;
    }
    $order_mobile = normalizeMobile($order['billing']['phone'] ?? '');
    if($order_mobile !== $mobile) {
        sendMessage($chat_id, '❗ شماره موبایل وارد شده با سفارش مطابقت ندارد.');
        return;
    }
    $product_id = $order['line_items'][0]['product_id'] ?? null;
    $product_name = $order['line_items'][0]['name'] ?? '';
    if(!$product_id) {
        sendMessage($chat_id, 'محصولی برای این سفارش یافت نشد.');
        return;
    }
    $stmt2 = dbQuery("SELECT * FROM product_messages WHERE product_id = ? ORDER BY step_number ASC", [$product_id]);
    $messages = $stmt2->fetchAll();
    if(!$messages) {
        sendMessage($chat_id, 'آموزشی برای این محصول ثبت نشده است.');
        return;
    }
    // بررسی وجود لایسنس برای این سفارش
    $license_exists = false;
    $license_key = null;
    if($order_id) {
        foreach($order['line_items'] as $item) {
            $item_product_id = $item['product_id'] ?? null;
            $item_variation_id = $item['variation_id'] ?? 0;
            if($item_product_id && $item_variation_id) {
                $stmt_lic = dbQuery("SELECT license_key FROM licenses WHERE product_id = ? AND variation_id = ? AND assigned_to_order = ? AND assigned_to_user = ?", [$item_product_id, $item_variation_id, $order_id, $mobile]);
                if($lic = $stmt_lic->fetch()) {
                    $license_exists = true;
                    $license_key = $lic['license_key'];
                    break;
                }
            }
        }
    }
    
    if($license_exists) {
        sendMessage($chat_id, "🎉 تبریک بابت خرید {$product_name}!\n\nلایسنس در پیام بالا/قبلی براتون ارسال شد لطفا کپی کنید\n✅ آموزش در حال ارسال است...");
    } else {
        sendMessage($chat_id, "🎉 تبریک بابت خرید {$product_name}!\nمنتظر باش تا ادمین برات لایسنس رو ارسال بکنه\n✅ آموزش در حال ارسال است...");
    }
    foreach($messages as $msg) { sendTrainingMessage($chat_id, $msg); sleep(1); }
    sendMessage($chat_id, '✅ آموزش به پایان رسید. برای سوالات بیشتر از پشتیبانی استفاده کنید.');
    dbQuery("INSERT INTO orders (user_id, product_id, mobile) VALUES (?, ?, ?)", [$chat_id, $product_id, $mobile]);
    dbQuery("UPDATE users SET score = score + 10 WHERE telegram_id = ?", [$chat_id]);
    // نمایش منوی اصلی پس از پایان آموزش
    $stmt_admin = dbQuery("SELECT telegram_id FROM admins WHERE telegram_id = ?", [$chat_id]);
    $is_admin = ($chat_id == ADMIN_ID) || ($stmt_admin->fetch());
    showStartMenu($chat_id, $is_admin);
}

function logAllProductsAndVariations() {
    $products = getProductsFromApi();
    $result = ["products" => []];
    foreach($products as $p) {
        $item = [
            "id" => $p["id"],
            "name" => $p["name"],
            "variations" => []
        ];
        $vars = getProductVariationsFromApi($p["id"]);
        if(is_array($vars)) {
            foreach($vars as $v) {
                $item["variations"][] = [
                    "id" => $v["id"] ?? null,
                    "attributes" => $v["attributes"] ?? [],
                    "price" => $v["price"] ?? null
                ];
            }
        }
        $result["products"][] = $item;
    }
    file_put_contents(__DIR__ . "/products_variations_log.json", json_encode($result, JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
} 